/*!************************************************************************
 * FILE :         clAudioSource.h
 * SW-COMPONENT:  Audio Engine
 * DESCRIPTION:
 *
 * Each AudioSource in the system is represented by an instance of AudioSource
 * or by a subclass of AudioSource.
 * AudioSource instances are used and controlled by the AudioSourceController.
 *
 * The individual subclasses of AudioSources implement the source-specific
 * communication with the corresponding MW components.
 *
 * Each AudioSource exposes an interface to the AudioSourceController with
 * the methods:
 * - vOn(...)     start source (player) and start demute-ramp
 * - vOff(...)    start mute-ramp and stop source (player)
 * - vPause(...)  start mute-ramp and pause source (player)
 * - bIsMuted(...)
 * - vReset(...)  stop player and mute source
 *
 * The AudioSource notifies the AudioSourceController via Callback-methods:
 * - vOn_Done()    the demute-ramp was started
 * - vOff_Done()...the mute-ramp has finished, player was stopped
 * - vPause_Done() the mute-ramp has finished, player was paused
 *
 * The AudioSource can have the states: (please see the state-charts for details)
 * - Off
 * - Pause
 * - On        demute-ramp was finished. Source is playing
 * - RampUp    AudioSource has started the demute-ramp
 * - RampDown  AudioSource has started the mute-ramp
 *
 * A statemachine ensures that requests are performed following the state-
 * transition rules.
 *
 * Sequence is:
 * - AudioSourceController calls vOn, vOff or vPause
 * - clAudioSource delegates request to StateMachine
 * - StateMachine applies rule-checks and might invoke vMW_On, vMW_Off, vMW_Pause
 * - vMW_On, vMW_Off, vMW_Pause do the communication with the MW
 *   Individual SubClasses of clAudioSouce may overwrite this methods to
 *   establish a specific MW communcation.
 * - Each implementation of vMW_On, vMW_Off, vMW_Pause has to confirm the operation
 *   asynchronously by calling vMW_OnDone, vMW_OffDone, vMW_PauseDone
 * - vMW_OnDone, vMW_OffDone, vMW_PauseDone will delegate the trigger
 *   again to the StateMachine.
 * - The StateMachine applies again the rules and invokes one of the methods:
 *   vNotifyOnDone,  vNotifyOffDone and  vNotifyPauseDone
 *   in order to inform the AudioSourceController
 *
 * TASKS TO IMPLEMENT A NEW AUDIOSOURCE:
 * - Provide a subclass of clAudioSource
 * - Overwrite the methods: vMW_On, vMW_Off, vMW_Pause
 *   Implement within these methods the specific communication with MW
 * - Ensure that these methods confirm the operation by invoking:
 *   vMW_OnDone, vMW_OffDone, vMW_PauseDone
 *
 *
 * AUTHOR:        CM-DI/PJ-VW34 Steinle
 * COPYRIGHT:     (c) 2006 Blaupunkt Werke
 *************************************************************************/
#ifndef _AUDIOSOURCE_H_
#define _AUDIOSOURCE_H_

#include "InterfaceAudioStack.h"
#include "AudioStack/clStackRules.h"
//#include "Audio_SM/Audio_SM_Trace_defines.h"
//#include "cliccaobserver.h"

//#define TRACE_WITHOUT_OSAL
//#define USE_DEBUG_INTERFACE
//#include "hmicca_pif.h"
namespace AudioStack
{

class clAudioSource;
class clSrcState;
class clAudioSMEngine;


struct SourceID
{
   AudioSources::enAudioSources enSourceClass;
   tU16                          u16SubSource;
   SourceID(AudioSources::enAudioSources srcClass, tU16 subSource) :enSourceClass(srcClass),u16SubSource(subSource){}
   //Avoid this constructor, but we need it because of stl map operator[]
   SourceID() : enSourceClass(static_cast<AudioSources::enAudioSources>(0)), u16SubSource(0) {}
   bool operator<( const SourceID & rhs ) const
   {
      return (
            ((((tU32)enSourceClass  )<< 16) | (tU32)  u16SubSource) <
            ((((tU32)rhs.enSourceClass)<< 16) | (tU32)rhs.u16SubSource) );
   }

   bool operator==(const SourceID& rhs) const
            {
      return (
            (rhs.enSourceClass == enSourceClass) &&
            (rhs.u16SubSource == u16SubSource) );
            }

   bool operator!=(const SourceID& rhs) const
               {
      return (
            (rhs.enSourceClass != enSourceClass) ||
            (rhs.u16SubSource != u16SubSource) );
               }

   SourceID& operator =(const SourceID& o)
   {
      enSourceClass = (o.enSourceClass);
      u16SubSource = (o.u16SubSource);
      return *this;
   }
};



/**
 * \brief
 * clAudioSourceObserver provides an abstract interface to identify
 * an acknowledgement for source switch sequences
 */
class clAudioSourceObserver
{
public:
   virtual ~clAudioSourceObserver(){};
   /**
   * vOn_done is called from the corresponding source implementation or base class,
   * when the ON-command has been acknowledged by the midlleware
   */
   virtual tVoid vOn_done(clAudioSource*)=0;
   /**
   * vOff_done is called from the corresponding source implementation or base class,
   * when the OFF-command has been acknowledged by the midlleware
   */
   virtual tVoid vOff_done(clAudioSource*)=0;
   /**
   * vPause_done is called from the corresponding source implementation or base class,
   * when the PAUSE-command has been acknowledged by the midlleware
   */
   virtual tVoid vPause_done(clAudioSource*)=0;
   /**
   * vInit_started is called from the corresponding source implementation or base class,
   * when the an initialisation sequence has been started
   */
   virtual tVoid vInit_started(clAudioSource*)=0;
};

/**
 * \brief
 * Each AudioSource in the system is represented by an instance of AudioSource
 * or by a subclass of AudioSource.
 * AudioSource instances are used and controlled by the AudioSourceController.
 *
 * The individual subclasses of AudioSources implement the source-specific
 * communication with the corresponding MW components.
 *
 * Each AudioSource exposes an interface to the AudioSourceController with
 * the methods:
 * - vOn(...)     start source (player) and start demute-ramp
 * - vOff(...)    start mute-ramp and stop source (player)
 * - vPause(...)  start mute-ramp and pause source (player)
 * - bIsMuted(...)
 * - vReset(...)  stop player and mute source
 *
 * The AudioSource notifies the AudioSourceController via Callback-methods:
 * - vOn_Done()    the demute-ramp was started
 * - vOff_Done()...the mute-ramp has finished, player was stopped
 * - vPause_Done() the mute-ramp has finished, player was paused
 *
 * The AudioSource can have the states: (please see the state-charts for details)
 * - Off
 * - Pause
 * - On        demute-ramp was finished. Source is playing
 * - RampUp    AudioSource has started the demute-ramp
 * - RampDown  AudioSource has started the mute-ramp
 *
 * A statemachine ensures that requests are performed following the state-
 * transition rules.
 *
 * Sequence is:
 * - AudioSourceController calls vOn, vOff or vPause
 * - clAudioSource delegates request to StateMachine
 * - StateMachine applies rule-checks and might invoke vMW_On, vMW_Off, vMW_Pause
 * - vMW_On, vMW_Off, vMW_Pause do the communication with the MW
 *   Individual SubClasses of clAudioSouce may overwrite this methods to
 *   establish a specific MW communcation.
 * - Each implementation of vMW_On, vMW_Off, vMW_Pause has to confirm the operation
 *   asynchronously by calling vMW_OnDone, vMW_OffDone, vMW_PauseDone
 * - vMW_OnDone, vMW_OffDone, vMW_PauseDone will delegate the trigger
 *   again to the StateMachine.
 * - The StateMachine applies again the rules and invokes one of the methods:
 *   vNotifyOnDone,  vNotifyOffDone and  vNotifyPauseDone
 *   in order to inform the AudioSourceController
 *
 * TASKS TO IMPLEMENT A NEW AUDIOSOURCE:
 * - Provide a subclass of clAudioSource
 * - Overwrite the methods: vMW_On, vMW_Off, vMW_Pause
 *   Implement within these methods the specific communication with MW
 * - Ensure that these methods confirm the operation by invoking:
 *   vMW_OnDone, vMW_OffDone, vMW_PauseDone
 */
class clAudioSource
{
public:

   typedef enum{
      On,
      Pause,
      Off
   } enSourceActivity;

   typedef enum {
      not_available
      , available
      , unkonwn
   } enSourceAvailability;

   typedef enum {
      newmedia
      , samemedia
      , nomedia
      , temperature
      , voltage
      , error
      , no_content
   } enSourceAvailabilityReason;

   friend class clSrcState;
   friend class clAudioSourceFactory;

   virtual ~clAudioSource();
   virtual tBool bCCACallback(tU32 u32msgcode, tU32 u32msgparam1, tU32 msgparam2);
   clAudioSource(SourceID enSrcId);

   virtual tVoid vMW_CCAResponse(tS32 extSourceId, enSourceActivity activity);

   /**
    * save the pointer to the object, which wants to observe the audio source switch sequence
    */
   tVoid vSetObserver(clAudioSourceObserver* poCallback){m_pclAudioSourceObserver=poCallback;}

   /**
    * availability of source changed
    */
   virtual tVoid vSourceAvailablilityChange(enSourceAvailability availability, enSourceAvailabilityReason availabilityReason);
   virtual tVoid vSourceAvailablilityChange();

   tVoid vOn();
   tVoid vOff    (SourceID u8PossibleNextSource);
   tVoid vPause  (SourceID u8PossibleNextSource);
   tVoid vWaitFor();

   /**
    * save the given data as userdata in a member
    */
   tVoid vSetUserData(tU32 u32UserData);

   /**
    * returns the data stored as userdata in a member
    */
   tU32 u32GetUserData(){return m_u32UserData;}

   /**
    * returns the audio source index stored as possible next source in a member
    */
   SourceID u8GetNextSource(){return m_possibleNextSource;}
   tBool bIsActive();
   tBool bIsOff();
   virtual tVoid vReset();

   virtual tVoid vNotifyOnDone();
   virtual tVoid vNotifyOffDone(); // can be overwritten by the corresponding source to reset the internal states
   virtual tVoid vNotifyPauseDone();
   virtual tVoid vNotifyInitStarted();

   /**
    * returns the string name of this audio source implementation
    */
   virtual const tChar *pacGetName(){return m_pacName;}
   SourceID sGetId();
//   tU8 u8GetNameIndex();
   const tChar* pacGetState();

   clStackRules::allow_t allowOnTop(clAudioSource *src);
   clStackRules::actions_t getPushAction(tU8 index);
   clStackRules::actions_t getPopAction(tU8 index);
   clStackRules::type_t getType();
   clStackRules::group_t getGroup();
   tU16 getAvailabilityTimeout() { return m_StackRule.getAvailabilityTimeout(); }
   clStackRules::registration_t  getRegistrationMode() { return m_StackRule.getRegistration(); }
   //int getNextBgSource(int index);
   clSrcState* pclGetState();
   tS32 getExternalId();

   virtual tVoid vPrint();

   //static AudioSources::enAudioSources u8getAudioSourceForHsi(tU8 enHSIAudioSource);

protected:
   // Audio Source State Machine
   tVoid Initialize(const tChar *pacName, SourceID enSrcId);
   /**
    * holds the current state of this audio source
    */
   clSrcState* m_pclState;


   clAudioSource(const tChar *pacName, SourceID enSrcId);

   /**
    * holds the string name of this audio source
    */
  const tChar *m_pacName;

   /**
    * holds the id of this audio source
    */
  SourceID m_SrcId;

   /**
    * holds the possible next source set by the member function
    */
   SourceID m_possibleNextSource;
   /**
    * Note! m_u32UserData is currently used for preset-data only. If you need it for
    * further data, you will have to apply a bit-mask to separate the data!
    */
   tU32 m_u32UserData;


public:
   // internal IF for MW CCA Communication
   virtual tVoid vMW_On();
   virtual tVoid vMW_Off(SourceID possibleNextSource);
   virtual tVoid vMW_Pause(SourceID possibleNextSource);
   virtual tVoid vMW_Init();

   //internal IF to ensure that the source is ready for activation
   virtual enSourceAvailability enIsSourceAvailable();

   // check play cmd counter to allow/disallow stop command
   tBool bPreCheckStateChange2Off();
   inline tVoid vResetPlayCmdCounter(){m_u32PlayCmdCounter = 0;};
protected:
   // internal IF for MW CCA Communication
   tVoid vMW_OnDone();
   tVoid vMW_OffDone();
   tVoid vMW_PauseDone();

   // to be called when a source finished to prepare and a play to FC-Audio is possible
   tVoid vMW_InitDone();

   // to be called when a source received a Confirmation from FC-Audio for a Play-Request
   // this is required to differentiate between the states: rampUpRequesting and rampUp
   // during RampUpRequesting, a new stop-request has to be confirmed manually, as FC-Audio will
   // not confirm.
   tVoid vMW_RequestConfirmed();

   // to be called when a required mw-service for a source becomes available
   // this is required to re-sync the source (finish the rampDown or restart)
   tVoid vMW_ServiceAvailable();

   // to be called when a required mw-service for a source becomes unavailable
   // currently not required. for future use only.
   tVoid vMW_ServiceUnavailable();

   /**
    * starts timer that will set source to unavailable if it is
    * in unkown state
    */
   tVoid vWatchdogStart_Availability();

   //**************************
   //****      MEMBERS
   //**************************
   tU32 m_u32PlayCmdCounter;
   /**
    * m_StackRules holds the rules for the order of sources on the stack
    */
   clStackRules m_StackRule;
   /*
    * Typically AudioSOurce Controller
    */
   clAudioSourceObserver* m_pclAudioSourceObserver;
   /**
    * actual availability state
    */
   enSourceAvailability m_availability;

   /**
    * reason for availability
    */
   enSourceAvailabilityReason m_availabilityReason;

private:

   static tVoid* vStart_ThreadedTimer(void* callBack);

};

}//namespace

#endif // _AUDIOSOURCE_H_
